"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createSigner = exports.fetchSignerType = exports.storeSignerType = exports.loadSignerType = void 0;
const SCWSigner_1 = require("./scw/SCWSigner");
const WalletLinkSigner_1 = require("./walletlink/WalletLinkSigner");
const error_1 = require("../core/error");
const provider_1 = require("../util/provider");
const ScopedLocalStorage_1 = require("../util/ScopedLocalStorage");
const SIGNER_TYPE_KEY = 'SignerType';
const storage = new ScopedLocalStorage_1.ScopedLocalStorage('CBWSDK', 'SignerConfigurator');
function loadSignerType() {
    return storage.getItem(SIGNER_TYPE_KEY);
}
exports.loadSignerType = loadSignerType;
function storeSignerType(signerType) {
    storage.setItem(SIGNER_TYPE_KEY, signerType);
}
exports.storeSignerType = storeSignerType;
async function fetchSignerType(params) {
    const { communicator, metadata } = params;
    listenForWalletLinkSessionRequest(communicator, metadata).catch(() => { });
    const request = {
        id: crypto.randomUUID(),
        event: 'selectSignerType',
        data: params.preference,
    };
    const { data } = await communicator.postRequestAndWaitForResponse(request);
    return data;
}
exports.fetchSignerType = fetchSignerType;
function createSigner(params) {
    const { signerType, metadata, communicator, updateListener } = params;
    switch (signerType) {
        case 'scw':
            return new SCWSigner_1.SCWSigner({
                metadata,
                updateListener,
                communicator,
            });
        case 'walletlink':
            return new WalletLinkSigner_1.WalletLinkSigner({
                metadata,
                updateListener,
            });
        case 'extension': {
            const injectedSigner = (0, provider_1.getCoinbaseInjectedSigner)();
            if (!injectedSigner) {
                throw error_1.standardErrors.rpc.internal('injected signer not found');
            }
            return injectedSigner;
        }
    }
}
exports.createSigner = createSigner;
async function listenForWalletLinkSessionRequest(communicator, metadata) {
    await communicator.onMessage(({ event }) => event === 'WalletLinkSessionRequest');
    // temporary walletlink signer instance to handle WalletLinkSessionRequest
    // will revisit this when refactoring the walletlink signer
    const walletlink = new WalletLinkSigner_1.WalletLinkSigner({
        metadata,
    });
    // send wallet link session to popup
    communicator.postMessage({
        event: 'WalletLinkUpdate',
        data: { session: walletlink.getSession() },
    });
    // wait for handshake to complete
    await walletlink.handshake();
    // send connected status to popup
    communicator.postMessage({
        event: 'WalletLinkUpdate',
        data: { connected: true },
    });
}
