import type { CaptchaWidgetType } from '@clerk/types';
interface RenderOptions {
    /**
     * Every widget has a sitekey. This sitekey is associated with the corresponding widget configuration and is created upon the widget creation.
     */
    sitekey: string;
    /**
     * Controls whether the widget should automatically retry to obtain a token if it did not succeed.
     * The default is auto, which will retry automatically. This can be set to never to disable retry upon failure.
     */
    retry?: 'auto' | 'never';
    /**
     * When retry is set to auto, retry-interval controls the time between retry attempts in milliseconds.
     * Value must be a positive integer less than 900000.
     * @default 8000
     */
    'retry-interval'?: number;
    /**
     * Automatically refreshes the token when it expires.
     * Can take auto, manual or never.
     * @default 'auto'
     */
    'refresh-expired'?: 'auto' | 'manual' | 'never';
    /**
     * A JavaScript callback invoked upon success of the challenge.
     * The callback is passed a token that can be validated.
     * @param token string
     */
    callback?: (token: string) => void;
    /**
     * A JavaScript callback invoked when there is an error (e.g. network error or the challenge failed)
     * @param errorCode string
     */
    'error-callback'?: (errorCode: string) => void;
    /**
     * A JavaScript callback invoked when a given client/browser is not supported by the widget.
     */
    'unsupported-callback'?: () => boolean;
    /**
     * Appearance controls when the widget is visible.
     * It can be always (default), execute, or interaction-only.
     * Refer to Appearance Modes for more information:
     * https://developers.cloudflare.com/turnstile/get-started/client-side-rendering/#appearance-modes
     * @default 'always'
     */
    appearance?: 'always' | 'execute' | 'interaction-only';
}
interface Turnstile {
    execute: (container?: string | HTMLElement | null, params?: RenderOptions) => void;
    render: (container?: string | HTMLElement | null, params?: RenderOptions) => string;
    remove: (widgetId: string) => void;
    reset: (widgetId: string) => void;
}
declare global {
    export interface Window {
        turnstile: Turnstile;
    }
}
export declare const shouldRetryTurnstileErrorCode: (errorCode: string) => boolean;
export declare const getTunstileToken: (captchaOptions: {
    siteKey: string;
    scriptUrl: string;
    widgetType: CaptchaWidgetType;
    invisibleSiteKey: string;
}) => Promise<{
    captchaToken: string;
    captchaWidgetTypeUsed: string;
}>;
export {};
